/*
  wiring_digital.c - digital input and output functions
  Part of Arduino - http://www.arduino.cc/

  Copyright (c) 2005-2006 David A. Mellis

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General
  Public License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
  Boston, MA  02111-1307  USA

  Modified 28 September 2010 by Mark Sproul
*/

#define ARDUINO_MAIN
#include "wiring_private.h"
#include "pins_arduino.h"

void pinMode(uint8_t pin, uint8_t mode)
{
	uint8_t bit = digitalPinToBitMask(pin);
	uint8_t port = digitalPinToPort(pin);
	volatile uint8_t *reg, *out;

	if (port == NOT_A_PIN) return;

	// JWS: can I let the optimizer do this?
	reg = portModeRegister(port);
	out = portOutputRegister(port);

	if (mode == INPUT) { 
		uint8_t oldSREG = SREG;
                cli();
		*reg &= ~bit;
		*out &= ~bit;
		SREG = oldSREG;
	} else if (mode == INPUT_PULLUP) {
		uint8_t oldSREG = SREG;
                cli();
		*reg &= ~bit;
		*out |= bit;
		SREG = oldSREG;
	} else {
		uint8_t oldSREG = SREG;
                cli();
		*reg |= bit;
		SREG = oldSREG;
	}
}

// Forcing this inline keeps the callers from having to push their own stuff
// on the stack. It is a good performance win and only takes 1 more byte per
// user than calling. (It will take more bytes on the 168.)
//
// But shouldn't this be moved into pinMode? Seems silly to check and do on
// each digitalread or write.
//
// Mark Sproul:
// - Removed inline. Save 170 bytes on atmega1280
// - changed to a switch statment; added 32 bytes but much easier to read and maintain.
// - Added more #ifdefs, now compiles for atmega645
//
//static inline void turnOffPWM(uint8_t timer) __attribute__ ((always_inline));
//static inline void turnOffPWM(uint8_t timer)
static void turnOffPWM(uint8_t timer)
{
	switch (timer)
	{
		#if defined(TCCR1A) && defined(COM1A1)
		case TIMER1A:   cbi(TCCR1A, COM1A1);    break;
		#endif
		#if defined(TCCR1A) && defined(COM1B1)
		case TIMER1B:   cbi(TCCR1A, COM1B1);    break;
		#endif
		#if defined(TCCR1A) && defined(COM1C1)
		case TIMER1C:   cbi(TCCR1A, COM1C1);    break;
		#endif
		
		#if defined(TCCR2) && defined(COM21)
		case  TIMER2:   cbi(TCCR2, COM21);      break;
		#endif
		
		#if defined(TCCR0A) && defined(COM0A1)
		case  TIMER0A:  cbi(TCCR0A, COM0A1);    break;
		#endif
		
		#if defined(TCCR0A) && defined(COM0B1)
		case  TIMER0B:  cbi(TCCR0A, COM0B1);    break;
		#endif
		#if defined(TCCR2A) && defined(COM2A1)
		case  TIMER2A:  cbi(TCCR2A, COM2A1);    break;
		#endif
		#if defined(TCCR2A) && defined(COM2B1)
		case  TIMER2B:  cbi(TCCR2A, COM2B1);    break;
		#endif
		
		#if defined(TCCR3A) && defined(COM3A1)
		case  TIMER3A:  cbi(TCCR3A, COM3A1);    break;
		#endif
		#if defined(TCCR3A) && defined(COM3B1)
		case  TIMER3B:  cbi(TCCR3A, COM3B1);    break;
		#endif
		#if defined(TCCR3A) && defined(COM3C1)
		case  TIMER3C:  cbi(TCCR3A, COM3C1);    break;
		#endif

		#if defined(TCCR4A) && defined(COM4A1)
		case  TIMER4A:  cbi(TCCR4A, COM4A1);    break;
		#endif					
		#if defined(TCCR4A) && defined(COM4B1)
		case  TIMER4B:  cbi(TCCR4A, COM4B1);    break;
		#endif
		#if defined(TCCR4A) && defined(COM4C1)
		case  TIMER4C:  cbi(TCCR4A, COM4C1);    break;
		#endif			
		#if defined(TCCR4C) && defined(COM4D1)
		case TIMER4D:	cbi(TCCR4C, COM4D1);	break;
		#endif			
			
		#if defined(TCCR5A)
		case  TIMER5A:  cbi(TCCR5A, COM5A1);    break;
		case  TIMER5B:  cbi(TCCR5A, COM5B1);    break;
		case  TIMER5C:  cbi(TCCR5A, COM5C1);    break;
		#endif
	}
}

void digitalWrite(uint8_t pin, uint8_t val)
{
	uint8_t timer = digitalPinToTimer(pin);
	uint8_t bit = digitalPinToBitMask(pin);
	uint8_t port = digitalPinToPort(pin);
	volatile uint8_t *out;

	if (port == NOT_A_PIN) return;

	// If the pin that support PWM output, we need to turn it off
	// before doing a digital write.
	if (timer != NOT_ON_TIMER) turnOffPWM(timer);

	out = portOutputRegister(port);

	uint8_t oldSREG = SREG;
	cli();

	if (val == LOW) {
		*out &= ~bit;
	} else {
		*out |= bit;
	}

	SREG = oldSREG;
}

int digitalRead(uint8_t pin)
{
	uint8_t timer = digitalPinToTimer(pin);
	uint8_t bit = digitalPinToBitMask(pin);
	uint8_t port = digitalPinToPort(pin);

	if (port == NOT_A_PIN) return LOW;

	// If the pin that support PWM output, we need to turn it off
	// before getting a digital reading.
	if (timer != NOT_ON_TIMER) turnOffPWM(timer);

	if (*portInputRegister(port) & bit) return HIGH;
	return LOW;
}



void pwmAtomicWriteWord(volatile unsigned char *p, unsigned int val)
{
	unsigned char _o_sreg = SREG;

	cli();
	*(p + 1) = (unsigned char)(val >> 8);
	_NOP(); _NOP(); _NOP();
	*p = (unsigned char)val;
	SREG = _o_sreg;
}

//	- For timer0/2 fmode has no effect
void pwmMode(unsigned char pin, unsigned char wmode, unsigned char fmode, unsigned char dband)
{
	volatile unsigned char *pTCCRX = 0;

	unsigned char timer = digitalPinToTimer(pin);

    switch(timer)
    {
    case TIMER0A:
		//pTCCRX = &TCCR0B;
        if( wmode & 0x10 )
            TCCR0A = (TCCR0A | _BV(COM0A0));
        else
            TCCR0A = (TCCR0A & ~_BV(COM0A0));
        break;
    case TIMER0B:
		//pTCCRX = &TCCR0B;
        if( wmode & 0x10 )
            TCCR0A = (TCCR0A | _BV(COM0B0));
        else
            TCCR0A = (TCCR0A & ~_BV(COM0B0));
        break;

    case TIMER1A:
		pTCCRX = &TCCR1B;
        if( wmode & 0x10 )
            TCCR1A = (TCCR1A | _BV(COM1A0));
        else
            TCCR1A = (TCCR1A & ~_BV(COM1A0));
        break;
    case TIMER1B:
		pTCCRX = &TCCR1B;
        if( wmode & 0x10 )
            TCCR1A = (TCCR1A | _BV(COM1B0));
        else
            TCCR1A = (TCCR1A & ~_BV(COM1B0));
        break;

    case TIMER2A:
		//pTCCRX = &TCCR2B;
        if( wmode & 0x10 )
            TCCR2A = (TCCR2A | _BV(COM2A0));
        else
            TCCR2A = (TCCR2A & ~_BV(COM2A0));
        break;
    case TIMER2B:
		//pTCCRX = &TCCR2B;
        if( wmode & 0x10 )
            TCCR2A = (TCCR2A | _BV(COM2B0));
        else
            TCCR2A = (TCCR2A & ~_BV(COM2B0));
        break;

    case TIMER3A:
		pTCCRX = &TCCR3B;
        if( wmode & 0x10 )
            TCCR3A = (TCCR3A | _BV(COM3A0));
        else
            TCCR3A = (TCCR3A & ~_BV(COM3A0));
        break;
    case TIMER3B:
		pTCCRX = &TCCR3B;
        if( wmode & 0x10 )
            TCCR3A = (TCCR3A | _BV(COM3B0));
        else
            TCCR3A = (TCCR3A & ~_BV(COM3B0));
        break;
    case TIMER3C:
		pTCCRX = &TCCR3B;
        if( wmode & 0x10 )
            TCCR3A = (TCCR3A | _BV(COM3C0));
        else
            TCCR3A = (TCCR3A & ~_BV(COM3C0));
        break;

    case TIMER4A:
		pTCCRX = &TCCR4B;
        if( wmode & 0x10 )
            TCCR4A = (TCCR4A | _BV(COM4A0));
        else
            TCCR4A = (TCCR4A & ~_BV(COM4A0));
        break;
    case TIMER4B:
		pTCCRX = &TCCR4B;
        if( wmode & 0x10 )
            TCCR4A = (TCCR4A | _BV(COM4B0));
        else
            TCCR4A = (TCCR4A & ~_BV(COM4B0));
        break;
    case TIMER4C:
		pTCCRX = &TCCR4B;
        if( wmode & 0x10 )
            TCCR4A = (TCCR4A | _BV(COM4C0));
        else
            TCCR4A = (TCCR4A & ~_BV(COM4C0));
        break;

    case TIMER5A:
		pTCCRX = &TCCR5B;
        if( wmode & 0x10 )
            TCCR5A = (TCCR5A | _BV(COM5A0));
        else
            TCCR5A = (TCCR5A & ~_BV(COM5A0));
        break;
    case TIMER5B:
		pTCCRX = &TCCR5B;
        if( wmode & 0x10 )
            TCCR5A = (TCCR5A | _BV(COM5B0));
        else
            TCCR5A = (TCCR5A & ~_BV(COM5B0));
        break;
    case TIMER5C:
		pTCCRX = &TCCR5B;
        if( wmode & 0x10 )
            TCCR5A = (TCCR5A | _BV(COM5C0));
        else
            TCCR5A = (TCCR5A & ~_BV(COM5C0));
        break;
    }

	if(pTCCRX == 0) return;

	if((fmode & 0x7f) == PWM_FREQ_SLOW) {
		*pTCCRX = (*pTCCRX & 0xf8) | PWM_FREQ_SLOW;	// prescale = 1024 (slowest mode)
	} else if((fmode & 0x7f) == PWM_FREQ_FAST) {
		*pTCCRX = (*pTCCRX & 0xf8) | PWM_FREQ_FAST; // prescale = 1 (fastest mode)
	} else if ((fmode & 0x7f) == PWM_FREQ_NORMAL) {
		*pTCCRX = (*pTCCRX & 0xf8) | PWM_FREQ_NORMAL;	// prescale = 64 (default)
	}
}

// enhanced PWM settings
// Function:
//	- set PWM frequency (unit: Hz), return maximum duty cycle 
// Note: 
//	- only PWM Timer1/3/4/5 support frequency update
unsigned int pwmFrequency(unsigned char pin, unsigned long fhz)
{
	unsigned long icrx = 0;
	unsigned char csxs = 0;
	unsigned char boost = 0;
	volatile unsigned char *pICRX = 0;
    unsigned long maxFreq = F_CPU;

	unsigned char timer = digitalPinToTimer(pin);

    if(timer == TIMER0A || timer == TIMER0B || timer == TIMER2A || timer == TIMER2B )
        return 0x0UL;

    switch(timer)
    {
    case TIMER1A:
    case TIMER1B:
        sbi(TCCR1B, WGM13);
        sbi(TCCR1B, WGM12);
        sbi(TCCR1A, WGM11);
        cbi(TCCR1A, WGM10);
		pICRX = &ICR1L;
		csxs = TCCR1B & 0x7;
        break;

    case TIMER3A:
    case TIMER3B:
    case TIMER3C:
        sbi(TCCR3B, WGM33);
        sbi(TCCR3B, WGM32);
        sbi(TCCR3A, WGM31);
        cbi(TCCR3A, WGM30);
		pICRX = &ICR3L;
		csxs = TCCR3B & 0x7;
        break;

    case TIMER4A:
    case TIMER4B:
    case TIMER4C:
        sbi(TCCR4B, WGM43);
        sbi(TCCR4B, WGM42);
        sbi(TCCR4A, WGM41);
        cbi(TCCR4A, WGM40);
		pICRX = &ICR4L;
		csxs = TCCR4B & 0x7;
        break;

    case TIMER5A:
    case TIMER5B:
    case TIMER5C:
        sbi(TCCR5B, WGM53);
        sbi(TCCR5B, WGM52);
        sbi(TCCR5A, WGM51);
        cbi(TCCR5A, WGM50);
		pICRX = &ICR5L;
		csxs = TCCR5B & 0x7;
        break;
    }


	if(pICRX == 0) return 0xff;

    if(boost)
    {
        maxFreq <<= 2;
    }

    switch( csxs )
    {
    case 1:
        break;
    case 2:
        maxFreq >>= 3;
        break;
    case 3:
        maxFreq >>= 6;
        break;
    case 4:
        maxFreq >>= 8;
        break;
    case 5:
        maxFreq >>= 10;
        break;
    }

    icrx = (maxFreq / fhz);
    if(icrx >= 65535u)
    {
        icrx = 65535u;
    }
	
	pwmAtomicWriteWord(pICRX, (unsigned int)icrx);

	return icrx;
}

// Function:
//	- return frequency (in Hz) by give PWM resolution (bits width of duty)
// Note: 
//	- timer0/2 works in FPWM mode, pwm frequency is fixed by given mode
//	- timer1/3/4/5 works in PCPWM mode, means frequency reduced by a half
unsigned long pwmResolution(unsigned char pin, unsigned char resBits)
{
	unsigned char csxs = 0;
	unsigned char boost = 0;
	unsigned long freq = 0x0UL;
    unsigned long maxFreq = F_CPU;
    unsigned long bitMask;

	unsigned char timer = digitalPinToTimer(pin);

    if(timer == TIMER0A || timer == TIMER0B || timer == TIMER2A || timer == TIMER2B )
        return 0x0UL;

    switch(timer)
    {
    case TIMER1A:
    case TIMER1B:
		csxs = TCCR1B & 0x7;
        break;
    case TIMER3B:
    case TIMER3C:
    case TIMER3A:
		csxs = TCCR3B & 0x7;
        break;
    case TIMER4A:
    case TIMER4B:
    case TIMER4C:
		csxs = TCCR4B & 0x7;
        break;
    case TIMER5C:
    case TIMER5B:
    case TIMER5A:
		csxs = TCCR5B & 0x7;
        break;
    }
	
    if(boost)
    {
        maxFreq <<= 2;
    }

    switch( csxs )
    {
    case 1:
        break;
    case 2:
        maxFreq >>= 3;
        break;
    case 3:
        maxFreq >>= 6;
        break;
    case 4:
        maxFreq >>= 8;
        break;
    case 5:
        maxFreq >>= 10;
        break;
    }
    
    bitMask = 1;
    bitMask <<= resBits;
    freq = maxFreq / bitMask;

	// update pwm frequency
	pwmFrequency(pin, freq);

	return freq;
}


// Exntenced PWM output
// Note: you can keep on use analogWrite() for compatible purpose!
void pwmWrite(unsigned char pin, unsigned int val)
{
	// We need to make sure the PWM output is enabled for those pins
	// that support it, as we turn it off when digitally reading or
	// writing with them.  Also, make sure the pin is in output mode
	// for consistenty with Wiring, which doesn't require a pinMode
	// call for the analog output pins.
	pinMode(pin, OUTPUT);
	if (val == 0)
	{
		digitalWrite(pin, LOW);
	}
	else if (val == 255)
	{
		digitalWrite(pin, HIGH);
	}
	else
	{
		switch(digitalPinToTimer(pin))
		{
			case TIMER0A:
				// connect pwm to pin on timer 0, channel A
				sbi(TCCR0A, COM0A1);
				OCR0A = val; // set pwm duty
				break;
			case TIMER0B:
				// connect pwm to pin on timer 0, channel B
				sbi(TCCR0A, COM0B1);
				OCR0B = val; // set pwm duty
				break;

			case TIMER1A:
				// connect pwm to pin on timer 1, channel A
				sbi(TCCR1A, COM1A1);
				OCR1A = val; // set pwm duty
				break;
			case TIMER1B:
				// connect pwm to pin on timer 1, channel B
				sbi(TCCR1A, COM1B1);
				OCR1B = val; // set pwm duty
				break;

			case TIMER1C:
				// connect pwm to pin on timer 1, channel B
				sbi(TCCR1A, COM1C1);
				OCR1C = val; // set pwm duty
				break;

			case TIMER2A:
				// connect pwm to pin on timer 2, channel A
				sbi(TCCR2A, COM2A1);
				OCR2A = val; // set pwm duty
				break;

			case TIMER2B:
				// connect pwm to pin on timer 2, channel B
				sbi(TCCR2A, COM2B1);
				OCR2B = val; // set pwm duty
				break;

			case TIMER3A:
				// connect pwm to pin on timer 3, channel A
				sbi(TCCR3A, COM3A1);
				OCR3A = val; // set pwm duty
				break;

			case TIMER3B:
				// connect pwm to pin on timer 3, channel B
				sbi(TCCR3A, COM3B1);
				OCR3B = val; // set pwm duty
				break;

			case TIMER3C:
				// connect pwm to pin on timer 3, channel C
				sbi(TCCR3A, COM3C1);
				OCR3C = val; // set pwm duty
				break;

			case TIMER4A:
				//connect pwm to pin on timer 4, channel A
				sbi(TCCR4A, COM4A1);
				#if defined(COM4A0)		// only used on 32U4
				cbi(TCCR4A, COM4A0);
				#endif
				OCR4A = val;	// set pwm duty
				break;
			
			case TIMER4B:
				// connect pwm to pin on timer 4, channel B
				sbi(TCCR4A, COM4B1);
				OCR4B = val; // set pwm duty
				break;

			case TIMER4C:
				// connect pwm to pin on timer 4, channel C
				sbi(TCCR4A, COM4C1);
				OCR4C = val; // set pwm duty
				break;

			case TIMER5A:
				// connect pwm to pin on timer 5, channel A
				sbi(TCCR5A, COM5A1);
				OCR5A = val; // set pwm duty
				break;

			case TIMER5B:
				// connect pwm to pin on timer 5, channel B
				sbi(TCCR5A, COM5B1);
				OCR5B = val; // set pwm duty
				break;

			case TIMER5C:
				// connect pwm to pin on timer 5, channel C
				sbi(TCCR5A, COM5C1);
				OCR5C = val; // set pwm duty
				break;

			case NOT_ON_TIMER:
			default:
				if (val < 128) {
					digitalWrite(pin, LOW);
				} else {
					digitalWrite(pin, HIGH);
				}
		}
	}
}


